# YarnAI VPS Deployment Guide

Complete guide to deploy YarnAI on your JaguarPC VPS with a custom domain.

---

## Prerequisites

- VPS with Ubuntu 20.04+ or Debian 11+
- SSH access to your VPS
- Domain name registered (e.g., yarnai.app)
- API keys ready (OpenAI, ElevenLabs)

---

## Quick Start (5 Steps)

### Step 1: Register Domain

1. Go to [Namecheap](https://namecheap.com), [Cloudflare](https://cloudflare.com), or [Google Domains](https://domains.google)
2. Register `yarnai.app` or your preferred domain (~$12-15/year)
3. Add DNS A record: `@` → Your VPS IP address
4. Add DNS A record: `www` → Your VPS IP address
5. Wait 5-30 minutes for DNS propagation

### Step 2: Download Deployment Package

Download `yarnai_vps_deploy.zip` from the Files panel and extract it.

### Step 3: Upload to VPS

```bash
# From your local machine
scp -r yarnai_vps_deploy/* root@YOUR_VPS_IP:/root/yarnai_deploy/
```

### Step 4: Configure Environment

SSH into your VPS and edit the environment file:

```bash
ssh root@YOUR_VPS_IP
cd /root/yarnai_deploy
cp .env.production .env
nano .env
```

Fill in your API keys:
- `OPENAI_API_KEY` - From OpenAI dashboard
- `ELEVENLABS_API_KEY` - From ElevenLabs dashboard  
- `GOOGLE_CLOUD_VISION_API_KEY` - From Google Cloud Console
- `DATABASE_URL` - See Database Options below
- `SESSION_SECRET` - Generate with `openssl rand -hex 32`

### Step 5: Run Deployment

```bash
chmod +x deploy.sh
DOMAIN=yarnai.app ./deploy.sh
```

The script will:
- Install Node.js 20, Nginx, PM2, Certbot
- Configure firewall (ports 22, 80, 443)
- Set up SSL certificate (Let's Encrypt)
- Start the application with PM2
- Configure auto-restart on reboot

---

## Database Options

### Option A: Managed PostgreSQL (Recommended)

Use [Neon](https://neon.tech) - free tier available:

1. Create account at neon.tech
2. Create new project "yarnai"
3. Copy connection string to `DATABASE_URL`

### Option B: PostgreSQL on VPS

```bash
# Install PostgreSQL
apt install postgresql postgresql-contrib

# Create database
sudo -u postgres createuser yarnai
sudo -u postgres createdb yarnai -O yarnai
sudo -u postgres psql -c "ALTER USER yarnai PASSWORD 'your-secure-password';"

# Update DATABASE_URL
DATABASE_URL=postgresql://yarnai:your-secure-password@localhost:5432/yarnai
```

---

## Directory Structure After Deployment

```
/var/www/yarnai/
├── dist/                 # Compiled application
│   └── index.js         # Server entry point
├── node_modules/        # Dependencies
├── package.json
├── .env                 # Environment variables
└── ecosystem.config.cjs # PM2 configuration

/var/log/yarnai/
├── error.log           # Application errors
├── output.log          # Standard output
└── combined.log        # All logs

/etc/nginx/sites-available/
└── yarnai              # Nginx configuration

/etc/letsencrypt/live/yarnai.app/
├── fullchain.pem       # SSL certificate
└── privkey.pem         # SSL private key
```

---

## Common Commands

### Application Management

```bash
# Check status
pm2 status

# View logs
pm2 logs yarnai

# Restart application
pm2 restart yarnai

# Stop application
pm2 stop yarnai

# Monitor resources
pm2 monit
```

### Nginx Management

```bash
# Test configuration
nginx -t

# Reload configuration
systemctl reload nginx

# View access logs
tail -f /var/log/nginx/access.log

# View error logs
tail -f /var/log/nginx/error.log
```

### SSL Certificate

```bash
# Check certificate status
certbot certificates

# Renew manually
certbot renew

# Test renewal
certbot renew --dry-run
```

---

## Updating the Application

When you have a new version:

```bash
# 1. Upload new files
scp -r new_dist/* root@YOUR_VPS_IP:/var/www/yarnai/dist/

# 2. Restart application
ssh root@YOUR_VPS_IP "pm2 restart yarnai"
```

Or create an update script:

```bash
#!/bin/bash
cd /var/www/yarnai
pm2 stop yarnai
# Pull new files or copy them
pm2 start yarnai
```

---

## Troubleshooting

### App not responding

```bash
# Check if app is running
pm2 status

# Check logs for errors
pm2 logs yarnai --lines 100

# Restart app
pm2 restart yarnai
```

### SSL certificate issues

```bash
# Check certificate
certbot certificates

# Force renewal
certbot renew --force-renewal

# Restart nginx
systemctl restart nginx
```

### Database connection issues

```bash
# Test connection
psql $DATABASE_URL -c "SELECT 1"

# Check PostgreSQL status
systemctl status postgresql
```

### Port conflicts

```bash
# Check what's using port 3000
lsof -i :3000

# Kill process if needed
kill -9 <PID>
```

---

## Security Checklist

- [ ] SSH key authentication enabled (disable password login)
- [ ] Firewall configured (only 22, 80, 443 open)
- [ ] SSL certificate installed
- [ ] Environment variables secured (chmod 600 .env)
- [ ] Regular security updates enabled
- [ ] PM2 running as non-root user

---

## Backup Strategy

### Database Backup

```bash
# Manual backup
pg_dump $DATABASE_URL > backup_$(date +%Y%m%d).sql

# Automated daily backup (add to crontab)
0 2 * * * pg_dump $DATABASE_URL > /backups/yarnai_$(date +\%Y\%m\%d).sql
```

### Application Backup

```bash
# Backup app directory
tar -czf yarnai_backup_$(date +%Y%m%d).tar.gz /var/www/yarnai
```

---

## Performance Tuning

### PM2 Cluster Mode

The ecosystem.config.cjs is configured to use all CPU cores:

```javascript
instances: 'max',
exec_mode: 'cluster'
```

### Nginx Caching

Static assets are cached for 30 days. Adjust in nginx.conf if needed.

### Memory Limits

PM2 will restart the app if it exceeds 500MB:

```javascript
max_memory_restart: '500M'
```

---

## Support

If you encounter issues:

1. Check PM2 logs: `pm2 logs yarnai`
2. Check Nginx logs: `tail -f /var/log/nginx/error.log`
3. Verify environment variables are set correctly
4. Ensure DNS is pointing to correct IP
5. Confirm SSL certificate is valid

---

## Quick Reference Card

| Task | Command |
|------|---------|
| Check app status | `pm2 status` |
| View logs | `pm2 logs yarnai` |
| Restart app | `pm2 restart yarnai` |
| Check SSL | `certbot certificates` |
| Renew SSL | `certbot renew` |
| Test nginx | `nginx -t` |
| Reload nginx | `systemctl reload nginx` |
